﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using Microsoft.Win32;
using System.IO;
using System.Windows.Markup;

namespace OcdToXamlConverter {

	public class Ocd9FileReader : OcdBaseFileReader {

		TFileHeader header;
		List<TBaseSym> symbols;
		List<TElement> objects;
		List<CustomParameterString> strings;
		List<CustomColor> colors;

		public Ocd9FileReader() {
			CleanUp();
		}

		public override void CleanUp() {
			header = null;
			symbols = null;
			objects = null;
			strings = null;
			colors = null;
			IsLoaded = false;
		}

		public override bool LoadFile(Stream stream) {
			CleanUp();
			bool result;
			header = new TFileHeader();
			result = header.Load(stream);
			if (!result) {
				CleanUp();
				stream.Close();
				return false;
			}
			symbols = ReadSymbols(stream);
			objects = ReadObjects(stream);
			strings = ReadStrings(stream);
			colors = LoadColors(strings);
			stream.Close();
			IsLoaded = true;
			return true;
		}

		public override int GetMainVersion() {
			if (!IsLoaded) {
				return 0;
			}
			return header.Version;
		}

		public override int GetSubVersion() {
			if (!IsLoaded) {
				return 0;
			}
			return header.Subversion;
		}

		public override double GetScale() {
			if (!IsLoaded) {
				return 0;
			}
			CustomParameterString scaleString = GetString(1039);
			if (scaleString == null) {
				return 0;
			}
			string scale = scaleString["m"];
			if (scale == null || scale == "") {
				return 0;
			}
			double d = 0;
			if (!double.TryParse(scale, out d)) {
				return 0;
			}
			return d;
		}

		private CustomParameterString GetString(int code) {
			for (int i = 0; i < strings.Count; i++) {
				if (strings[i].Type == code) {
					return strings[i];
				}
			}
			return null;
		}

		public override FrameworkElement CreateMap() {
			if (IsLoaded == false) {
				return null;
			}
			Canvas canvas = new Canvas();
			Rect bounds = new Rect();
			bool assignedRect = false;
			for (int i = 0; i < objects.Count; i++) {
				Rect rect;
				CreateObject(objects[i], canvas, out rect);
				if (rect.Width > 0 && rect.Height > 0) {
					if (assignedRect) {
						bounds = Rect.Union(bounds, rect);
					} else {
						bounds = rect;
						assignedRect = true;
					}
				}
			}
			Canvas map = new Canvas();
			map.Background = Brushes.White;
			map.Width = bounds.Width;
			map.Height = bounds.Height;
			Canvas.SetLeft(canvas, -bounds.Left);
			Canvas.SetTop(canvas, -bounds.Top);
			map.Children.Add(canvas);
			return map;
		}

		private bool CreateObject(TElement element, Canvas parent, out Rect rect) {
			bool substractStokeWidth = false;
			rect = Rect.Empty;
			TBaseSym symbol = GetSymbolByNum(element.Sym);
			if (symbol == null) {
				return false;
			}
			if (symbol.Status == 2) {
				return true;
			}
			if (symbol is TPointSym) {
				TPointSym pointSymbol = (TPointSym)symbol;
				FrameworkElement obj = CreatePointSymbol(pointSymbol.Elts, substractStokeWidth);
				if (obj == null) {
					return false;
				}
				if (element.Poly.Length <= 0) {
					return false;
				}
				rect = GetExtent(pointSymbol.Elts);
				Canvas.SetZIndex(obj, GetColorDepth(GetMainColorNum(pointSymbol.Elts)));
				Canvas.SetLeft(obj, GetLength(element.Poly[0].XPos));
				Canvas.SetTop(obj, -GetLength(element.Poly[0].YPos));
				if (element.Ang != 0) {
					Transform tranform = new RotateTransform(-((double)element.Ang) * 0.1);
					obj.LayoutTransform = tranform;
					rect.Transform(tranform.Value);
				}
				parent.Children.Add(obj);
				rect.Offset(GetLength(element.Poly[0].XPos), -GetLength(element.Poly[0].YPos));
				return true;
			}
			if (symbol is TLineSym) {
				TLineSym lineSymbol = (TLineSym)symbol;
				if (element.Poly.Length == 0) {
					return false;
				}
				bool drawnMainPath = false;
				OcdPathHelper pathHelper = new OcdPathHelper(element.Poly);
				if (pathHelper.Length == 0) {
					return true;
				}
				if (lineSymbol.MainGap == 0 && lineSymbol.SecGap == 0) {
					if ((lineSymbol.LineEnds & 2) == 2 && (lineSymbol.DistFromStart != 0 || lineSymbol.DistToEnd != 0)) {
						PathFigure pathFigure = pathHelper.GetPointedPath(GetLength(lineSymbol.LineWidth), GetLength(lineSymbol.DistFromStart), GetLength(lineSymbol.DistToEnd));
						PathGeometry geometry = new PathGeometry(new PathFigure[] { pathFigure });
						System.Windows.Shapes.Path path = new System.Windows.Shapes.Path();
						path.Data = geometry;
						path.Fill = new SolidColorBrush(GetColor(lineSymbol.LineColor));
						path.StrokeLineJoin = PenLineJoin.Bevel;
						if ((lineSymbol.LineEnds & 1) == 1) {
							path.StrokeLineJoin = PenLineJoin.Round;
						} else if ((lineSymbol.LineEnds & 4) == 4) {
							path.StrokeLineJoin = PenLineJoin.Miter;
						}
						Canvas.SetZIndex(path, GetColorDepth(lineSymbol.LineColor));
						parent.Children.Add(path);
						Pen pen = new Pen(Brushes.Black, GetLength(lineSymbol.LineWidth));
						rect = Rect.Union(rect, geometry.GetRenderBounds(pen));
					} else {
						PathFigure pathFigure = pathHelper.GetPathFigure();
						PathGeometry geometry = new PathGeometry(new PathFigure[] { pathFigure });
						System.Windows.Shapes.Path path = new System.Windows.Shapes.Path();
						path.Data = geometry;
						path.Stroke = new SolidColorBrush(GetColor(lineSymbol.LineColor));
						path.StrokeThickness = GetLength(lineSymbol.LineWidth);
						path.StrokeLineJoin = PenLineJoin.Bevel;
						if ((lineSymbol.LineEnds & 1) == 1) {
							path.StrokeLineJoin = PenLineJoin.Round;
						} else if ((lineSymbol.LineEnds & 4) == 4) {
							path.StrokeLineJoin = PenLineJoin.Miter;
						}
						Canvas.SetZIndex(path, GetColorDepth(lineSymbol.LineColor));
						parent.Children.Add(path);
						Pen pen = new Pen(Brushes.Black, GetLength(lineSymbol.LineWidth));
						rect = Rect.Union(rect, geometry.GetRenderBounds(pen));
					}
					drawnMainPath = true;
				}
				if (lineSymbol.MainSymbol.Length > 0 || lineSymbol.MainGap != 0 || lineSymbol.SecGap != 0) {
					List<OcdPathHelper> pathHelpers = pathHelper.SplitByCorner();
					double maxDist = 0;
					for (int i = 0; i < pathHelpers.Count; i++) {
						double totalLength = pathHelpers[i].Length;
						maxDist = Math.Max(maxDist, totalLength);
					}
					for (int i = 0; i < pathHelpers.Count; i++) {
						double mainLength = GetLength(lineSymbol.MainLength);
						double endLength = GetLength(lineSymbol.EndLength);
						double mainGap = GetLength(lineSymbol.MainGap);
						double secGap = GetLength(lineSymbol.SecGap);
						double midLength;
						double totalLength = pathHelpers[i].Length;
						double primSymDist = GetLength(lineSymbol.PrimSymDist);
						int num = -1;
						double strokeOffset = mainLength - endLength;
						bool hasDash = true;
						if (mainLength > 0) {
							midLength = totalLength - endLength * 2;
							if (midLength <= 0) {
								double mainGapFrac = mainGap / endLength;
								endLength = (totalLength - mainGap) / 2;//
								mainLength = totalLength;//
								num = 0;//
								strokeOffset = mainLength - endLength;
								secGap = 0;//
								if (GetLength(lineSymbol.EndLength) >= totalLength) {
									num = -1;//
									if (lineSymbol.MainSymbol.Length > 0 || Math.Abs(maxDist - totalLength) >= 0.5) {//
										hasDash = false;
									}
								}
							} else {
								double endLengthFrac = endLength / mainLength;
								num = (int)Math.Round((midLength - mainGap) / (mainLength + mainGap) + 0.1);//
								mainLength = (totalLength - (num + 1.0) * mainGap) / (2 * endLengthFrac + num) - 0.0001;//
								endLength = endLengthFrac * mainLength;
								strokeOffset = mainLength - endLength;
							}
						}
						if (secGap >= mainLength) {
							secGap = 0;//
						}
						if (Math.Abs(maxDist - totalLength) < 0.5 && lineSymbol.MinSym > -1 && num == -1) {//
							num = 0;
							if (endLength + (lineSymbol.nPrimSym - 1) * primSymDist > totalLength) {//
								endLength = Math.Max(0, (totalLength - (lineSymbol.nPrimSym - 1) * primSymDist) / 2);//
							}//
						}
						if (!drawnMainPath) {
							System.Windows.Shapes.Path path = new System.Windows.Shapes.Path();
							PathFigure pathFigure = pathHelpers[i].GetPathFigure();
							PathGeometry geometry = new PathGeometry(new PathFigure[] { pathFigure });
							path.Data = geometry;
							path.Stroke = new SolidColorBrush(GetColor(lineSymbol.LineColor));
							path.StrokeThickness = GetLength(lineSymbol.LineWidth);
							if (hasDash == true) {
								if (lineSymbol.SecGap > 0) {
									double firstLength = 0.5 * (mainLength - secGap);
									double firstGapLength = secGap;
									double secondGapLength = mainGap;
									path.StrokeDashArray = new DoubleCollection(new double[] { firstLength / GetLength(lineSymbol.LineWidth), firstGapLength / GetLength(lineSymbol.LineWidth), firstLength / GetLength(lineSymbol.LineWidth), secondGapLength / GetLength(lineSymbol.LineWidth) });
								} else {
									double gapLength = mainGap;
									path.StrokeDashArray = new DoubleCollection(new double[] { mainLength / GetLength(lineSymbol.LineWidth), gapLength / GetLength(lineSymbol.LineWidth) });
								}
								path.StrokeDashOffset = strokeOffset / GetLength(lineSymbol.LineWidth);//
							}
							if (i > 0) {
								path.StrokeStartLineCap = PenLineCap.Square;//
							}
							if (i < pathHelpers.Count - 1) {
								path.StrokeEndLineCap = PenLineCap.Square;//
							}
							path.StrokeLineJoin = PenLineJoin.Bevel;
							if ((lineSymbol.LineEnds & 1) == 1) {
								path.StrokeLineJoin = PenLineJoin.Round;
							} else if ((lineSymbol.LineEnds & 4) == 4) {
								path.StrokeLineJoin = PenLineJoin.Miter;
							}
							Canvas.SetZIndex(path, GetColorDepth(lineSymbol.LineColor));
							parent.Children.Add(path);
							Pen pen = new Pen(Brushes.Black, GetLength(lineSymbol.LineWidth));
							rect = Rect.Union(rect, geometry.GetRenderBounds(pen));
						}
						double distance = endLength + mainGap;
						for (int j = 0; j < num + 1; j++) {
							if (distance + (lineSymbol.nPrimSym - 1) * primSymDist > totalLength) {
								continue;
							}
							for (int k = 0; k < lineSymbol.nPrimSym; k++) {
								Point newPoint;
								double angle = 0;
								newPoint = pathHelpers[i].GetPointAtDistance(distance);
								angle = pathHelpers[i].GetAngleAtDistance(distance) / Math.PI * 180.0;
								FrameworkElement mainSymbol = CreatePointSymbol(lineSymbol.MainSymbol, substractStokeWidth);
								Transform transform = new RotateTransform(angle);
								mainSymbol.LayoutTransform = transform;
								Rect temp = GetExtent(lineSymbol.MainSymbol);
								temp.Transform(transform.Value);
								temp.Offset(newPoint.X, newPoint.Y);
								rect = Rect.Union(rect, temp);
								Canvas.SetLeft(mainSymbol, newPoint.X);
								Canvas.SetTop(mainSymbol, newPoint.Y);
								Canvas.SetZIndex(mainSymbol, GetColorDepth(GetMainColorNum(lineSymbol.MainSymbol)));
								parent.Children.Add(mainSymbol);
								distance += primSymDist;
							}
							distance += mainLength + mainGap - lineSymbol.nPrimSym * primSymDist;
						}
					}
				}
				if (lineSymbol.CornerSymbol.Length > 0) {
					for (int i = 0; i < pathHelper.OcdSegments.Count; i++) {
						Point newPoint;
						double angle;
						if (pathHelper.OcdSegments[i].IsStartCornerPoint) {
							newPoint = pathHelper.OcdSegments[i].Start;
							angle = pathHelper.OcdSegments[i].GetAngleAtStart(true) / Math.PI * 180.0;
						} else if (pathHelper.OcdSegments[i].IsEndCornerPoint && i == pathHelper.OcdSegments.Count - 1) {
							newPoint = pathHelper.OcdSegments[i].End;
							angle = pathHelper.OcdSegments[i].GetAngleAtEnd(true) / Math.PI * 180.0;
						} else {
							continue;
						}
						FrameworkElement cornerSymbol = CreatePointSymbol(lineSymbol.CornerSymbol, substractStokeWidth);
						Transform transform = new RotateTransform(angle);
						cornerSymbol.LayoutTransform = transform;
						Rect temp = GetExtent(lineSymbol.CornerSymbol);
						temp.Transform(transform.Value);
						temp.Offset(newPoint.X, newPoint.Y);
						rect = Rect.Union(rect, temp);
						Canvas.SetLeft(cornerSymbol, newPoint.X);
						Canvas.SetTop(cornerSymbol, newPoint.Y);
						Canvas.SetZIndex(cornerSymbol, GetColorDepth(GetMainColorNum(lineSymbol.CornerSymbol)));
						parent.Children.Add(cornerSymbol);
					}
				}
				if (lineSymbol.EndSymbol.Length > 0 && pathHelper.OcdSegments.Count > 0) {
					Point newPoint;
					double angle = pathHelper.OcdSegments[pathHelper.OcdSegments.Count - 1].GetAngleAtEnd(false) / Math.PI * 180.0;
					newPoint = pathHelper.End;
					FrameworkElement endSymbol = CreatePointSymbol(lineSymbol.EndSymbol, substractStokeWidth);
					Transform transform = new RotateTransform(angle);
					endSymbol.LayoutTransform = transform;
					Rect temp = GetExtent(lineSymbol.EndSymbol);
					temp.Transform(transform.Value);
					temp.Offset(newPoint.X, newPoint.Y);
					rect = Rect.Union(rect, temp);
					Canvas.SetLeft(endSymbol, newPoint.X);
					Canvas.SetTop(endSymbol, newPoint.Y);
					Canvas.SetZIndex(endSymbol, GetColorDepth(GetMainColorNum(lineSymbol.EndSymbol)));
					parent.Children.Add(endSymbol);
				}
				if (lineSymbol.StartSymbol.Length > 0 && pathHelper.OcdSegments.Count > 0) {
					Point newPoint;
					double angle = pathHelper.OcdSegments[0].GetAngleAtStart(false) / Math.PI * 180.0;
					newPoint = pathHelper.Start;
					FrameworkElement startSymbol = CreatePointSymbol(lineSymbol.StartSymbol, substractStokeWidth);
					Transform transform = new RotateTransform(angle);
					startSymbol.LayoutTransform = transform;
					Rect temp = GetExtent(lineSymbol.StartSymbol);
					temp.Transform(transform.Value);
					temp.Offset(newPoint.X, newPoint.Y);
					rect = Rect.Union(rect, temp);
					Canvas.SetLeft(startSymbol, newPoint.X);
					Canvas.SetTop(startSymbol, newPoint.Y);
					Canvas.SetZIndex(startSymbol, GetColorDepth(GetMainColorNum(lineSymbol.StartSymbol)));
					parent.Children.Add(startSymbol);
				}
				if (lineSymbol.DblMode != 0) {
					if (lineSymbol.DblFlags != 0) {
						PathFigure pathFigure = pathHelper.GetPathFigure();
						PathGeometry geometry = new PathGeometry(new PathFigureCollection(new PathFigure[] { pathFigure }));
						System.Windows.Shapes.Path path = new System.Windows.Shapes.Path();
						path.Stroke = new SolidColorBrush(GetColor(lineSymbol.DblFillColor));
						path.StrokeThickness = GetLength(lineSymbol.DblWidth);
						path.Data = geometry;
						Canvas.SetZIndex(path, GetColorDepth(lineSymbol.DblFillColor));
						parent.Children.Add(path);
						Pen pen = new Pen(Brushes.Black, GetLength(lineSymbol.DblWidth));
						rect = Rect.Union(rect, geometry.GetRenderBounds(pen));
					}
					if (lineSymbol.DblLeftWidth > 0) {
						PathFigure newFigure = pathHelper.GetShiftedPath(GetLength(lineSymbol.DblWidth / 2 + lineSymbol.DblLeftWidth / 2));
						PathGeometry newGeometry = new PathGeometry(new PathFigureCollection(new PathFigure[] { newFigure }));
						System.Windows.Shapes.Path path = new System.Windows.Shapes.Path();
						path.Stroke = new SolidColorBrush(GetColor(lineSymbol.DblLeftColor));
						path.StrokeThickness = GetLength(lineSymbol.DblLeftWidth);
						path.Data = newGeometry;
						if (lineSymbol.DblMode == 2 || lineSymbol.DblMode == 3) {
							path.StrokeDashArray = new DoubleCollection(new double[] { GetLength(lineSymbol.DblLength) / GetLength(lineSymbol.DblLeftWidth), GetLength(lineSymbol.DblGap) / GetLength(lineSymbol.DblLeftWidth) });
						}
						Canvas.SetZIndex(path, GetColorDepth(lineSymbol.DblLeftColor));
						parent.Children.Add(path);
						Pen pen = new Pen(Brushes.Black, GetLength(lineSymbol.DblLeftWidth));
						rect = Rect.Union(rect, newGeometry.GetRenderBounds(pen));
					}
					if (lineSymbol.DblRightWidth > 0) {
						PathFigure newFigure = pathHelper.GetShiftedPath(-GetLength(lineSymbol.DblWidth / 2 + lineSymbol.DblRightWidth / 2));
						PathGeometry newGeometry = new PathGeometry(new PathFigureCollection(new PathFigure[] { newFigure }));
						System.Windows.Shapes.Path path = new System.Windows.Shapes.Path();
						path.Stroke = new SolidColorBrush(GetColor(lineSymbol.DblRightColor));
						path.StrokeThickness = GetLength(lineSymbol.DblRightWidth);
						path.Data = newGeometry;
						if (lineSymbol.DblMode == 3) {
							path.StrokeDashArray = new DoubleCollection(new double[] { GetLength(lineSymbol.DblLength) / GetLength(lineSymbol.DblRightWidth), GetLength(lineSymbol.DblGap) / GetLength(lineSymbol.DblRightWidth) });
						}
						Canvas.SetZIndex(path, GetColorDepth(lineSymbol.DblRightColor));
						parent.Children.Add(path);
						Pen pen = new Pen(Brushes.Black, GetLength(lineSymbol.DblRightWidth));
						rect = Rect.Union(rect, newGeometry.GetRenderBounds(pen));
					}
				}
				return true;
			}
			if (symbol is TAreaSym) {
				TAreaSym areaSymbol = (TAreaSym)symbol;
				Brush brush;
				Color color;
				System.Windows.Shapes.Path path;
				Geometry geometry = GetAreaGeometry(element.Poly);
				rect = geometry.Bounds;
				if (areaSymbol.FillOn == true) {
					color = GetColor(areaSymbol.FillColor);
					brush = new SolidColorBrush(GetColor(areaSymbol.FillColor));
					path = new System.Windows.Shapes.Path();
					path.Fill = brush;
					path.Data = geometry;
					Canvas.SetZIndex(path, GetColorDepth(areaSymbol.FillColor));
					parent.Children.Add(path);
				}
				if (areaSymbol.HatchMode == 1 && areaSymbol.HatchLineWidth > 0) {
					HatchedArea hatchedArea;
					hatchedArea = new HatchedArea();
					hatchedArea.HatchLineWidth = GetLength(areaSymbol.HatchLineWidth);
					hatchedArea.HatchLineSpacing = GetLength(areaSymbol.HatchDist);
					hatchedArea.SpacingMode = SpacingMode.Absolute;
					hatchedArea.HatchColor = GetColor(areaSymbol.HatchColor);
					hatchedArea.FillTransform = new RotateTransform(-0.1 * (element.Ang + areaSymbol.HatchAngle1 - 90 * 10));
					hatchedArea.Data = geometry;
					Canvas.SetZIndex(hatchedArea, GetColorDepth(areaSymbol.HatchColor));
					parent.Children.Add(hatchedArea);
				}
				if (areaSymbol.HatchMode == 2 && areaSymbol.HatchLineWidth > 0) {
					HatchedArea hatchedArea;
					hatchedArea = new HatchedArea();
					hatchedArea.HatchLineWidth = GetLength(areaSymbol.HatchLineWidth);
					hatchedArea.HatchLineSpacing = GetLength(areaSymbol.HatchDist);
					hatchedArea.SpacingMode = SpacingMode.Absolute;
					hatchedArea.HatchColor = GetColor(areaSymbol.HatchColor);
					hatchedArea.FillTransform = new RotateTransform(-0.1 * (element.Ang + areaSymbol.HatchAngle1 - 90 * 10));
					hatchedArea.Data = geometry;
					Canvas.SetZIndex(hatchedArea, GetColorDepth(areaSymbol.HatchColor));
					parent.Children.Add(hatchedArea);
					hatchedArea = new HatchedArea();
					hatchedArea.HatchLineWidth = GetLength(areaSymbol.HatchLineWidth);
					hatchedArea.HatchLineSpacing = GetLength(areaSymbol.HatchDist);
					hatchedArea.SpacingMode = SpacingMode.Absolute;
					hatchedArea.HatchColor = GetColor(areaSymbol.HatchColor);
					hatchedArea.FillTransform = new RotateTransform(-0.1 * (element.Ang + areaSymbol.HatchAngle2 - 90 * 10));
					hatchedArea.Data = geometry;
					Canvas.SetZIndex(hatchedArea, GetColorDepth(areaSymbol.HatchColor));
					parent.Children.Add(hatchedArea);
				}
				if (areaSymbol.StructMode == 1 && areaSymbol.Elts.Length > 0) {
					TiledArea tiledArea;
					tiledArea = new TiledArea();
					tiledArea.TargetDrawing = CreateElementsGroup(areaSymbol.Elts, substractStokeWidth);
					tiledArea.FillTransform = new RotateTransform(-0.1 * areaSymbol.StructAngle);
					tiledArea.SpacingMode = SpacingMode.Absolute;
					tiledArea.HorizontalSpacing = GetLength(areaSymbol.StructWidth) * 1.08;
					tiledArea.VerticalSpacing = GetLength(areaSymbol.StructHeight) * 1.08;
					tiledArea.Data = geometry;
					Canvas.SetZIndex(tiledArea, GetColorDepth(GetMainColorNum(areaSymbol.Elts)));
					parent.Children.Add(tiledArea);
				}
				if (areaSymbol.StructMode == 2 && areaSymbol.Elts.Length > 0) {
					TiledArea tiledArea;
					tiledArea = new TiledArea();
					tiledArea.TargetDrawing = CreateElementsGroup(areaSymbol.Elts, substractStokeWidth);
					tiledArea.FillTransform = new RotateTransform(-0.1 * areaSymbol.StructAngle);
					tiledArea.SpacingMode = SpacingMode.Absolute;
					tiledArea.HorizontalSpacing = GetLength(areaSymbol.StructWidth) * 1.08;
					tiledArea.VerticalSpacing = GetLength(areaSymbol.StructHeight) * 1.08;
					tiledArea.Data = geometry;
					tiledArea.TileMode = CustomTileMode.Honeycomb;
					Canvas.SetZIndex(tiledArea, GetColorDepth(GetMainColorNum(areaSymbol.Elts)));
					parent.Children.Add(tiledArea);
				}
				return true;
			}
			if (symbol is TRectSym) {
				TRectSym rectSymbol = (TRectSym)symbol;
				Rectangle rectangle = new Rectangle();
				Brush brush = new SolidColorBrush(GetColor(rectSymbol.LineColor));
				double thickness = GetLength(rectSymbol.LineWidth);
				rectangle.Stroke = brush;
				rectangle.StrokeThickness = thickness;
				rectangle.RadiusX = GetLength(rectSymbol.Radius);
				rectangle.RadiusY = GetLength(rectSymbol.Radius);
				if (element.Poly.Length < 4) {
					return false;
				}
				double x = GetLength(element.Poly[3].XPos);
				double y = GetLength(element.Poly[3].YPos); ;
				double width = GetLength(element.Poly[2].XPos - element.Poly[0].XPos);
				double height = GetLength(element.Poly[2].YPos - element.Poly[0].YPos);
				if (width < 0) {
					x -= width;
					width = -width;
				}
				if (height < 0) {
					y -= height;
					height = -height;
				}
				x -= thickness / 2;
				y += thickness / 2;
				width += thickness;
				height += thickness;
				rectangle.Width = width;
				rectangle.Height = height;
				Canvas.SetZIndex(rectangle, GetColorDepth(rectSymbol.LineColor));
				Canvas.SetLeft(rectangle, x);
				Canvas.SetTop(rectangle, -y);
				parent.Children.Add(rectangle);
				rect = new Rect(x, -y, width, height);
				return true;
			}
			if (symbol is TTextSym) {
				TTextSym textSymbol = (TTextSym)symbol;
				string text = element.ExtraString;
				if (text.Length == 0) {
					return true;
				}
				if (element.Poly.Length <= 0) {
					return false;
				}
				if (text.StartsWith("\r")) {
					text = text.Substring(1);
				}
				if (text.StartsWith("\n")) {
					text = text.Substring(1);
				}
				Rect textRect;
				StackPanel stackPanel = new StackPanel();
				double fontSize = ((double)textSymbol.FontSize) / 10.0 * 96.0 / 72.0;
				double lineHeight = textSymbol.LineSpace * fontSize / 100.0;
				Brush textBrush = new SolidColorBrush(GetColor(textSymbol.FontColor));
				FontFamily font = new FontFamily(textSymbol.FontName);
				FontStyle style = FontStyles.Normal;
				FontWeight weight = FontWeights.Normal;
				TextAlignment alignment = TextAlignment.Left;
				if (textSymbol.Alignment == 1) {
					alignment = TextAlignment.Center;
				} else if (textSymbol.Alignment == 2) {
					alignment = TextAlignment.Right;
				}
				if (textSymbol.Weight == 700) {
					weight = FontWeights.Bold;
				}
				if (textSymbol.Italic == true) {
					style = FontStyles.Italic;
				}
				double topMargin = 0;
				string[] lines = text.Split(new string[] { "\r\n" }, StringSplitOptions.None);
				for (int i = 0; i < lines.Length; i++) {
					string lineText = lines[i];
					if (textSymbol.Tabs.Length > 0 && lineText.IndexOf("\t") >= 0) {
						string[] textFragments = lineText.Split(new char[] { '\t' });
						StackPanel linePanel = new StackPanel();
						linePanel.Orientation = Orientation.Horizontal;
						double lastWidth = 0;
						double currentColWidth = 0;
						double lastColumnWidth = GetLength(textSymbol.Tabs[0]);
						double currentMargin = 0;
						if (textSymbol.Tabs.Length > 1) {
							lastColumnWidth = GetLength(textSymbol.Tabs[textSymbol.Tabs.Length - 1] - textSymbol.Tabs[textSymbol.Tabs.Length - 2]);
						}
						int currentCol = 0;
						for (int j = 0; j < textFragments.Length; j++) {
							if (j > 0) {
								currentMargin = -lastWidth;
								do {
									if (currentCol == 0) {
										currentColWidth = GetLength(textSymbol.Tabs[0]);
									} else if (currentCol < textSymbol.Tabs.Length) {
										currentColWidth = GetLength(textSymbol.Tabs[currentCol] - textSymbol.Tabs[currentCol - 1]);
									} else {
										currentColWidth = lastColumnWidth;
									}
									currentCol++;
									currentMargin += currentColWidth;
								} while (currentMargin < 0);
							} else {
								currentMargin = 0;
							}
							TextBlock textBlock = new TextBlock();
							textBlock.Text = textFragments[j];
							textBlock.Foreground = textBrush;
							textBlock.FontFamily = font;
							textBlock.FontStyle = style;
							textBlock.LineHeight = lineHeight;
							textBlock.FontSize = fontSize;
							textBlock.LineStackingStrategy = LineStackingStrategy.BlockLineHeight;
							textBlock.FontWeight = weight;
							textBlock.TextAlignment = alignment;
							textBlock.TextWrapping = TextWrapping.Wrap;
							textBlock.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
							lastWidth = textBlock.DesiredSize.Width;
							textBlock.Margin = new Thickness(currentMargin, topMargin, 0, 0);
							linePanel.Children.Add(textBlock);
						}
						stackPanel.Children.Add(linePanel);
					} else {
						TextBlock textBlock = new TextBlock();
						textBlock.Text = lineText.Replace("\t", "");
						textBlock.Foreground = textBrush;
						textBlock.FontFamily = font;
						textBlock.FontStyle = style;
						textBlock.LineHeight = lineHeight;
						textBlock.FontSize = fontSize;
						textBlock.LineStackingStrategy = LineStackingStrategy.BlockLineHeight;
						textBlock.FontWeight = weight;
						textBlock.TextAlignment = alignment;
						textBlock.TextWrapping = TextWrapping.Wrap;
						textBlock.Margin = new Thickness(0, topMargin, 0, 0);
						stackPanel.Children.Add(textBlock);
					}
					if (textSymbol.LBOn != 0 && textSymbol.LBWidth > 0) {
						Rectangle lineRect = new Rectangle();
						lineRect.Height = GetLength(textSymbol.LBWidth);
						lineRect.Margin = new Thickness(0, GetLength(textSymbol.LBDist) - 0.2 * lineHeight, 0, 0.2 * lineHeight);
						lineRect.Fill = new SolidColorBrush(GetColor(textSymbol.LBColor));
						Canvas.SetZIndex(lineRect, -1);
						stackPanel.Children.Add(lineRect);
					}
					topMargin = GetLength(textSymbol.ParaSpace);
				}
				Size desiredSize = new Size();
				if (element.Poly.Length == 4) {
					double x = GetLength(element.Poly[3].XPos);
					double y = GetLength(element.Poly[3].YPos); ;
					double width = GetLength(element.Poly[1].XPos - element.Poly[0].XPos);
					double height = GetLength(element.Poly[2].YPos - element.Poly[1].YPos);
					if (width < 0) {
						x -= width;
						width = -width;
					}
					if (height < 0) {
						y -= height;
						height = -height;
					}
					stackPanel.Width = width;
					stackPanel.Height = height;
					textRect = new Rect(x, -y, width, height);
				} else {
					stackPanel.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
					desiredSize = stackPanel.DesiredSize;
					textRect = new Rect(GetLength(element.Poly[0].XPos), -GetLength(element.Poly[0].YPos) - lineHeight * 0.8, desiredSize.Width, desiredSize.Height);
				}
				rect = textRect;
				Canvas.SetZIndex(stackPanel, GetColorDepth(textSymbol.FontColor));
				if (textSymbol.Alignment == 1) {
					if (element.Poly.Length != 4) {
						rect.Offset(-desiredSize.Width / 2, 0);
					}
				} else if (textSymbol.Alignment == 2) {
					if (element.Poly.Length != 4) {
						rect.Offset(-desiredSize.Width, 0);
					}
				}
				Canvas.SetLeft(stackPanel, rect.Left);
				Canvas.SetTop(stackPanel, rect.Top);
				if (element.Ang != 0) {
					Transform tranform = new RotateTransform(-((double)element.Ang) * 0.1, 0, lineHeight * 0.8);
					stackPanel.RenderTransform = tranform;
					tranform = new RotateTransform(-((double)element.Ang) * 0.1, rect.Left, rect.Top + lineHeight * 0.8);
					rect.Transform(tranform.Value);
				}
				parent.Children.Add(stackPanel);
				return true;
			}
			if (symbol is TLTextSym) {
				TLTextSym lineTextSymbol = (TLTextSym)symbol;
				TextBlock textBlock;
				Brush brush = new SolidColorBrush(GetColor(lineTextSymbol.FontColor));
				double fontSize = ((double)lineTextSymbol.FontSize) / 10.0 * 96.0 / 72.0;
				string text = element.ExtraString;
				if (text.StartsWith("\r")) {
					text = text.Substring(1);
				}
				if (text.StartsWith("\n")) {
					text = text.Substring(1);
				}
				text = text.Replace("\t", "");//
				FontFamily font = new FontFamily(lineTextSymbol.FontName);
				FontWeight weight = FontWeights.Normal;
				FontStyle style = FontStyles.Normal;
				if (lineTextSymbol.Italic == true) {
					style = FontStyles.Italic;
				}
				if (lineTextSymbol.Weight == 700) {
					weight = FontWeights.Bold;
				}
				if (element.Poly.Length <= 0) {
					return false;
				}
				if (text == "") {
					return true;
				}
				double textLength;
				OcdPathHelper pathHelper = new OcdPathHelper(element.Poly);
				PathFigure pathFigure = pathHelper.GetPathFigure();
				double totalLength = pathHelper.Length;
				double distance = 0;
				for (int i = 0; i < text.Length; i++) {
					textBlock = new TextBlock();
					textBlock.FontSize = fontSize;
					textBlock.FontFamily = font;
					textBlock.FontWeight = weight;
					textBlock.FontStyle = style;
					textBlock.Text = text.Substring(i, 1);
					textBlock.Foreground = brush;
					textBlock.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
					textLength = textBlock.DesiredSize.Width;
					double textheight = fontSize;
					Point newPoint;
					double angle = 0;
					newPoint = pathHelper.GetPointAtDistance(distance);
					angle = pathHelper.GetAngleAtDistance(distance);
					newPoint.Offset(fontSize * Math.Sin(angle), -fontSize * Math.Cos(angle));
					Transform transform = new RotateTransform(angle / Math.PI * 180.0, 0, 0);
					textBlock.RenderTransform = transform;
					Rect temp = new Rect(new Point(0, 0), textBlock.DesiredSize);
					temp.Transform(transform.Value);
					temp.Offset(newPoint.X, newPoint.Y);
					rect = Rect.Union(rect, temp);
					Canvas.SetLeft(textBlock, newPoint.X);
					Canvas.SetTop(textBlock, newPoint.Y);
					Canvas.SetZIndex(textBlock, GetColorDepth(lineTextSymbol.FontColor));
					parent.Children.Add(textBlock);
					textLength *= (1.0 + lineTextSymbol.CharSpace / 100.0) / 1.1;//
					distance += textLength;
					if (distance > totalLength) {
						break;
					}
				}
				return true;
			}
			return false;
		}

		protected override int GetColorDepth(int num) {
			for (int i = 0; i < colors.Count; i++) {
				if (colors[i].Num == num) {
					return -(i + 1);
				}
			}
			return 0;
		}

		private List<TBaseSym> ReadSymbols(Stream stream) {
			List<TBaseSym> symbols = new List<TBaseSym>();
			long pos = stream.Seek((long)header.FirstSymBlk, SeekOrigin.Begin);
			while (pos > 0) {
				TSymbolBlock symBlock = new TSymbolBlock();
				symBlock.Load(stream);
				for (int i = 0; i < symBlock.FilePos.Length; i++) {
					if (symBlock.FilePos[i] == 0) {
						continue;
					}
					stream.Seek((long)symBlock.FilePos[i], SeekOrigin.Begin);
					TBaseSym symbol = TBaseSym.Create(stream);
					if (symbol == null) {
						continue;
					}
					symbols.Add(symbol);
				}
				pos = stream.Seek((long)symBlock.NextBlock, SeekOrigin.Begin);
			}
			return symbols;
		}

		private List<TElement> ReadObjects(Stream stream) {
			List<TElement> objects = new List<TElement>();
			long pos = stream.Seek((long)header.FirstIdxBlk, SeekOrigin.Begin);
			while (pos > 0) {
				TIndexBlock indexBlock = new TIndexBlock();
				indexBlock.Load(stream);
				for (int i = 0; i < indexBlock.IndexArr.Length; i++) {
					if (indexBlock.IndexArr[i].Len == 0 || indexBlock.IndexArr[i].Pos == 0 || indexBlock.IndexArr[i].Sym == 0 || indexBlock.IndexArr[i].Status != 1) {
						continue;
					}
					stream.Seek((long)indexBlock.IndexArr[i].Pos, SeekOrigin.Begin);
					TElement element = new TElement();
					element.Load(stream);
					objects.Add(element);
				}
				pos = stream.Seek((long)indexBlock.NextBlock, SeekOrigin.Begin);
			}
			return objects;
		}

		private List<CustomParameterString> ReadStrings(Stream stream) {
			List<CustomParameterString> strings = new List<CustomParameterString>();
			long pos = stream.Seek((long)header.FirstStIndexBlk, SeekOrigin.Begin);
			while (pos > 0) {
				TStringIndexBlock indexBlock = new TStringIndexBlock();
				indexBlock.Load(stream);
				for (int i = 0; i < indexBlock.Table.Length; i++) {
					if (indexBlock.Table[i].Pos == 0 || indexBlock.Table[i].Len == 0) {
						continue;
					}
					CustomParameterString param = new CustomParameterString();
					param.Load(stream, indexBlock.Table[i]);
					strings.Add(param);
				}
				pos = stream.Seek((long)indexBlock.FilePos, SeekOrigin.Begin);
			}
			return strings;
		}

		private List<CustomColor> LoadColors(List<CustomParameterString> strings) {
			List<CustomColor> colors = new List<CustomColor>();
			for (int i = 0; i < strings.Count; i++) {
				if (strings[i].Type == 9) {
					CustomColor color = new CustomColor();
					color.Load(strings[i]);
					colors.Add(color);
				}
			}
			return colors;
		}

		protected override Color GetColor(Int16 num) {
			for (int i = 0; i < colors.Count; i++) {
				if (colors[i].Num == num) {
					return colors[i].Color;
				}
			}
			return Colors.Transparent;
		}

		private TBaseSym GetSymbolByNum(int symbolNum) {
			for (int i = 0; i < symbols.Count; i++) {
				if (symbols[i].Sym == symbolNum) {
					return symbols[i];
				}
			}
			return null;
		}

		private class TFileHeader {

			public Int16 OCADMark;
			public byte FileType;
			public byte FileStatus;
			public Int16 Version;
			public Int16 Subversion;
			public Int32 FirstSymBlk;
			public Int32 FirstIdxBlk;
			public Int32 Res0;
			public Int32 Res1;
			public Int32 Res2;
			public Int32 InfoSize;
			public Int32 FirstStIndexBlk;
			public Int32 FileNamePos;
			public Int32 FileNameSize;
			public Int32 Res4;

			public bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream);
				OCADMark = br.ReadInt16();
				if (OCADMark != 0x0cad) {
					return false;
				}
				FileType = br.ReadByte();
				FileStatus = br.ReadByte();
				Version = br.ReadInt16();
				if (Version != 9 && Version != 10) {
					return false;
				}
				Subversion = br.ReadInt16();
				FirstSymBlk = br.ReadInt32();
				FirstIdxBlk = br.ReadInt32();
				Res0 = br.ReadInt32();
				Res1 = br.ReadInt32();
				Res2 = br.ReadInt32();
				InfoSize = br.ReadInt32();
				FirstStIndexBlk = br.ReadInt32();
				FileNamePos = br.ReadInt32();
				FileNameSize = br.ReadInt32();
				Res4 = br.ReadInt32();
				return true;
			}

		}

		private abstract class TBaseSym {

			public static TBaseSym Create(Stream stream) {
				TBaseSym symbol;
				long pos = stream.Position;
				stream.Seek(sizeof(Int32) * 2, SeekOrigin.Current);
				BinaryReader br = new BinaryReader(stream);
				byte otp = br.ReadByte();
				stream.Seek(pos, SeekOrigin.Begin);
				switch (otp) {
					case 1:
						symbol = new TPointSym();
						break;
					case 2:
						symbol = new TLineSym();
						break;
					case 3:
						symbol = new TAreaSym();
						break;
					case 4:
						symbol = new TTextSym();
						break;
					case 6:
						symbol = new TLTextSym();
						break;
					case 7:
						symbol = new TRectSym();
						break;
					default:
						return null;
				}
				symbol.Load(stream);
				return symbol;
			}

			public Int32 Size;
			public Int32 Sym;
			public byte Otp;
			public byte Flags;
			public bool Selected;
			public byte Status;
			public byte DrawingTool;
			public byte CsMode;
			public byte CsObjType;
			public byte CsCdFlags;
			public Int32 Extent;
			public Int32 FilePos;
			public Int16 Group;
			public Int16 nColors;
			public Int16[] Colors = new Int16[14];//?
			public string Description;
			public byte[] IconBits = new byte[484];

			public virtual bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream);
				Size = br.ReadInt32();
				Sym = br.ReadInt32();
				Otp = br.ReadByte();
				Flags = br.ReadByte();
				Selected = br.ReadBoolean();
				Status = br.ReadByte();
				DrawingTool = br.ReadByte();
				CsMode = br.ReadByte();
				CsObjType = br.ReadByte();
				CsCdFlags = br.ReadByte();
				Extent = br.ReadInt32();
				FilePos = br.ReadInt32();
				Group = br.ReadInt16();
				nColors = br.ReadInt16();
				for (int i = 0; i < Colors.Length; i++) {
					Colors[i] = br.ReadInt16();
				}
				byte[] chars = br.ReadBytes(32);
				string s = System.Text.ASCIIEncoding.ASCII.GetString(chars, 1, (int)chars[0]);
				Description = s;
				IconBits = br.ReadBytes(484);
				return true;
			}

		}

		private class TPointSym : TBaseSym {

			public Int16 DataSize;
			public Int16 Reserved;

			public override bool Load(Stream stream) {
				base.Load(stream);
				BinaryReader br = new BinaryReader(stream);
				DataSize = br.ReadInt16();
				Reserved = br.ReadInt16();
				int i = 0;
				List<TSymElt> elts = new List<TSymElt>();
				while (i < DataSize) {
					TSymElt elt = new TSymElt();
					elt.Load(stream);
					elts.Add(elt);
					i += elt.stnPoly + 2; //2 = size of TSymElt Header
				}
				Elts = elts.ToArray();
				return true;
			}

			//Custom Members

			public TSymElt[] Elts;

		}

		private class TLineSym : TBaseSym {

			public Int16 LineColor;
			public Int16 LineWidth;
			public UInt16 LineEnds;
			public Int16 DistFromStart;
			public Int16 DistToEnd;
			public Int16 MainLength;
			public Int16 EndLength;
			public Int16 MainGap;
			public Int16 SecGap;
			public Int16 EndGap;
			public Int16 MinSym;
			public Int16 nPrimSym;
			public Int16 PrimSymDist;
			public UInt16 DblMode;
			public UInt16 DblFlags;
			public Int16 DblFillColor;
			public Int16 DblLeftColor;
			public Int16 DblRightColor;
			public Int16 DblWidth;
			public Int16 DblLeftWidth;
			public Int16 DblRightWidth;
			public Int16 DblLength;
			public Int16 DblGap;
			public Int16[] DblRes = new Int16[3];
			public UInt16 DecMode;
			public Int16 DecLast;
			public Int16 DecRes;
			public Int16 FrColor;
			public Int16 FrWidth;
			public Int16 FrStyle;
			public Int16 PrimDSize;
			public Int16 SecDSize;
			public Int16 CornerDSize;
			public Int16 StartDSize;
			public Int16 EndDSize;
			public Int16 Reserved;

			public override bool Load(Stream stream) {
				base.Load(stream);
				BinaryReader br = new BinaryReader(stream);
				LineColor = br.ReadInt16();
				LineWidth = br.ReadInt16();
				LineEnds = br.ReadUInt16();
				DistFromStart = br.ReadInt16();
				DistToEnd = br.ReadInt16();
				MainLength = br.ReadInt16();
				EndLength = br.ReadInt16();
				MainGap = br.ReadInt16();
				SecGap = br.ReadInt16();
				EndGap = br.ReadInt16();
				MinSym = br.ReadInt16();
				nPrimSym = br.ReadInt16();
				PrimSymDist = br.ReadInt16();
				DblMode = br.ReadUInt16();
				DblFlags = br.ReadUInt16();
				DblFillColor = br.ReadInt16();
				DblLeftColor = br.ReadInt16();
				DblRightColor = br.ReadInt16();
				DblWidth = br.ReadInt16();
				DblLeftWidth = br.ReadInt16();
				DblRightWidth = br.ReadInt16();
				DblLength = br.ReadInt16();
				DblGap = br.ReadInt16();
				for (int j = 0; j < DblRes.Length; j++) {
					DblRes[j] = br.ReadInt16();
				}
				DecMode = br.ReadUInt16();
				DecLast = br.ReadInt16();
				DecRes = br.ReadInt16();
				FrColor = br.ReadInt16();
				FrWidth = br.ReadInt16();
				FrStyle = br.ReadInt16();
				PrimDSize = br.ReadInt16();
				SecDSize = br.ReadInt16();
				CornerDSize = br.ReadInt16();
				StartDSize = br.ReadInt16();
				EndDSize = br.ReadInt16();
				Reserved = br.ReadInt16();
				int i;
				List<TSymElt> elts;
				elts = new List<TSymElt>();
				i = 0;
				while (i < PrimDSize) {
					TSymElt elt = new TSymElt();
					elt.Load(stream);
					elts.Add(elt);
					i += elt.stnPoly + 2; //2 = size of TSymElt Header
				}
				MainSymbol = elts.ToArray();
				elts = new List<TSymElt>();
				i = 0;
				while (i < SecDSize) {
					TSymElt elt = new TSymElt();
					elt.Load(stream);
					elts.Add(elt);
					i += elt.stnPoly + 2; //2 = size of TSymElt Header
				}
				SecondarySymbol = elts.ToArray();
				elts = new List<TSymElt>();
				i = 0;
				while (i < CornerDSize) {
					TSymElt elt = new TSymElt();
					elt.Load(stream);
					elts.Add(elt);
					i += elt.stnPoly + 2; //2 = size of TSymElt Header
				}
				CornerSymbol = elts.ToArray();
				elts = new List<TSymElt>();
				i = 0;
				while (i < StartDSize) {
					TSymElt elt = new TSymElt();
					elt.Load(stream);
					elts.Add(elt);
					i += elt.stnPoly + 2; //2 = size of TSymElt Header
				}
				StartSymbol = elts.ToArray();
				elts = new List<TSymElt>();
				i = 0;
				while (i < EndDSize) {
					TSymElt elt = new TSymElt();
					elt.Load(stream);
					elts.Add(elt);
					i += elt.stnPoly + 2; //2 = size of TSymElt Header
				}
				EndSymbol = elts.ToArray();
				return true;
			}

			//Custom Members

			public TSymElt[] MainSymbol;
			public TSymElt[] SecondarySymbol;
			public TSymElt[] CornerSymbol;
			public TSymElt[] StartSymbol;
			public TSymElt[] EndSymbol;

		}

		private class TAreaSym : TBaseSym {

			public Int32 BorderSym;
			public Int16 FillColor;
			public Int16 HatchMode;
			public Int16 HatchColor;
			public Int16 HatchLineWidth;
			public Int16 HatchDist;
			public Int16 HatchAngle1;
			public Int16 HatchAngle2;
			public bool FillOn;
			public bool BorderOn;
			public Int16 StructMode;
			public Int16 StructWidth;
			public Int16 StructHeight;
			public Int16 StructAngle;
			public Int16 Res;
			public Int16 DataSize;

			public override bool Load(Stream stream) {
				base.Load(stream);
				BinaryReader br = new BinaryReader(stream);
				BorderSym = br.ReadInt32();
				FillColor = br.ReadInt16();
				HatchMode = br.ReadInt16();
				HatchColor = br.ReadInt16();
				HatchLineWidth = br.ReadInt16();
				HatchDist = br.ReadInt16();
				HatchAngle1 = br.ReadInt16();
				HatchAngle2 = br.ReadInt16();
				FillOn = br.ReadBoolean();
				BorderOn = br.ReadBoolean();
				StructMode = br.ReadInt16();
				StructWidth = br.ReadInt16();
				StructHeight = br.ReadInt16();
				StructAngle = br.ReadInt16();
				Res = br.ReadInt16();
				DataSize = br.ReadInt16();
				int i = 0;
				List<TSymElt> elts = new List<TSymElt>();
				while (i < DataSize) {
					TSymElt elt = new TSymElt();
					elt.Load(stream);
					elts.Add(elt);
					i += elt.stnPoly + 2; //2 = size of TSymElt Header
				}
				Elts = elts.ToArray();
				return true;
			}

			//Custom Members

			public TSymElt[] Elts;

		}

		private class TTextSym : TBaseSym {

			public string FontName;
			public Int16 FontColor;
			public Int16 FontSize;
			public Int16 Weight;
			public bool Italic;
			public byte CharSet;
			public Int16 CharSpace;
			public Int16 WordSpace;
			public Int16 Alignment;
			public Int16 LineSpace;
			public Int16 ParaSpace;
			public Int16 IndentFirst;
			public Int16 IndentOther;
			public Int16 nTabs;
			public Int32[] Tabs;
			public Int16 LBOn; //wordbool
			public Int16 LBColor;
			public Int16 LBWidth;
			public Int16 LBDist;
			public Int16 Res4;
			public byte FrMode;
			public byte FrLineStyle;
			public string Res2;
			public Int16 FrLeft;
			public Int16 FrBottom;
			public Int16 FrRight;
			public Int16 FrTop;
			public Int16 FrColor;
			public Int16 FrSize;
			public Int16 FrWeight;
			public UInt16 FrItalic;
			public Int16 FrOfX;
			public Int16 FrOfY;

			public override bool Load(Stream stream) {
				base.Load(stream);
				BinaryReader br = new BinaryReader(stream);
				string s;
				byte[] chars;
				chars = br.ReadBytes(32);
				s = System.Text.ASCIIEncoding.ASCII.GetString(chars, 1, (int)chars[0]);
				FontName = s;
				FontColor = br.ReadInt16();
				FontSize = br.ReadInt16();
				Weight = br.ReadInt16();
				Italic = br.ReadBoolean();
				CharSet = br.ReadByte();
				CharSpace = br.ReadInt16();
				WordSpace = br.ReadInt16();
				Alignment = br.ReadInt16();
				LineSpace = br.ReadInt16();
				ParaSpace = br.ReadInt16();
				IndentFirst = br.ReadInt16();
				IndentOther = br.ReadInt16();
				nTabs = br.ReadInt16();
				Tabs = new Int32[nTabs];
				int i;
				for (i = 0; i < Tabs.Length; i++) {
					Tabs[i] = br.ReadInt32();
				}
				while (i < 32) {
					br.ReadInt32();
					i++;
				}
				LBOn = br.ReadInt16();
				LBColor = br.ReadInt16();
				LBWidth = br.ReadInt16();
				LBDist = br.ReadInt16();
				Res4 = br.ReadInt16();
				FrMode = br.ReadByte();
				FrLineStyle = br.ReadByte();
				chars = br.ReadBytes(24);
				Res2 = "";
				FrLeft = br.ReadInt16();
				FrBottom = br.ReadInt16();
				FrRight = br.ReadInt16();
				FrTop = br.ReadInt16();
				FrColor = br.ReadInt16();
				FrSize = br.ReadInt16();
				FrWeight = br.ReadInt16();
				FrItalic = br.ReadUInt16();
				FrOfX = br.ReadInt16();
				FrOfY = br.ReadInt16();
				return true;
			}

		}

		private class TRectSym : TBaseSym {

			public Int16 LineColor;
			public Int16 LineWidth;
			public Int16 Radius;
			public UInt16 GridFlags;
			public Int16 CellWidth;
			public Int16 CellHeight;
			public Int16 ResGridLineColor;
			public Int16 ResGridLineWidth;
			public Int16 UnnumCells;
			public string UnnumText;
			public Int16 GridRes2;
			public string ResFontName;
			public Int16 ResFontColor;
			public Int16 ResFontSize;
			public Int16 ResWeight;
			public Int16 ResItalic; //wordbool
			public Int16 ResOfsX;
			public Int16 ResOfsY;

			public override bool Load(Stream stream) {
				base.Load(stream);
				BinaryReader br = new BinaryReader(stream);
				string s;
				byte[] chars;
				LineColor = br.ReadInt16();
				LineWidth = br.ReadInt16();
				Radius = br.ReadInt16();
				GridFlags = br.ReadUInt16();
				CellWidth = br.ReadInt16();
				CellHeight = br.ReadInt16();
				ResGridLineColor = br.ReadInt16();
				ResGridLineWidth = br.ReadInt16();
				UnnumCells = br.ReadInt16();
				chars = br.ReadBytes(4);
				s = System.Text.ASCIIEncoding.ASCII.GetString(chars, 1, (int)chars[0]);
				UnnumText = s;
				GridRes2 = br.ReadInt16();
				chars = br.ReadBytes(32);
				s = System.Text.ASCIIEncoding.ASCII.GetString(chars, 1, (int)chars[0]);
				ResFontName = s;
				ResFontColor = br.ReadInt16();
				ResFontSize = br.ReadInt16();
				ResWeight = br.ReadInt16();
				ResItalic = br.ReadInt16();
				ResOfsX = br.ReadInt16();
				ResOfsY = br.ReadInt16();
				return true;
			}

		}

		private class TLTextSym : TBaseSym {

			public string FontName;
			public Int16 FontColor;
			public Int16 FontSize;
			public Int16 Weight;
			public bool Italic;
			public byte CharSet;
			public Int16 CharSpace;
			public Int16 WordSpace;
			public Int16 Alignment;
			public Int16 FrMode;
			public string FrName;
			public Int16 FrColor;
			public Int16 FrSize;
			public Int16 FrWeight;
			public Int16 FrItalic;//wordbool;
			public Int16 FrOfX;
			public Int16 FrOfY;

			public override bool Load(Stream stream) {
				base.Load(stream);
				BinaryReader br = new BinaryReader(stream);
				string s;
				byte[] chars;
				chars = br.ReadBytes(32);
				s = System.Text.ASCIIEncoding.ASCII.GetString(chars, 1, (int)chars[0]);
				FontName = s;
				FontColor = br.ReadInt16();
				FontSize = br.ReadInt16();
				Weight = br.ReadInt16();
				Italic = br.ReadBoolean();
				CharSet = br.ReadByte();
				CharSpace = br.ReadInt16();
				WordSpace = br.ReadInt16();
				Alignment = br.ReadInt16();
				FrMode = br.ReadInt16();
				chars = br.ReadBytes(32);
				s = System.Text.ASCIIEncoding.ASCII.GetString(chars, 1, (int)chars[0]);
				FrName = s;
				FrColor = br.ReadInt16();
				FrSize = br.ReadInt16();
				FrWeight = br.ReadInt16();
				FrItalic = br.ReadInt16();
				FrOfX = br.ReadInt16();
				FrOfY = br.ReadInt16();
				return true;
			}

		}

		private class TIndexBlock {

			public Int32 NextBlock;
			public TIndex[] IndexArr = new TIndex[256];

			public bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream);
				NextBlock = br.ReadInt32();
				for (int i = 0; i < IndexArr.Length; i++) {
					TIndex index = new TIndex();
					index.Load(stream);
					IndexArr[i] = index;
				}
				return true;
			}

		}

		private class TIndex {

			public TCord LowerLeft;
			public TCord UpperRight;
			public Int32 Pos;
			public Int32 Len;
			public Int32 Sym;
			public byte ObjType;
			public byte Res1;
			public byte Status;
			public byte ViewType;
			public Int16 Color;
			public Int16 Res2;
			public Int16 ImpLayer;
			public Int16 Res3;

			public bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream);
				TCord cord;
				cord = new TCord();
				cord.Load(stream);
				LowerLeft = cord;
				cord = new TCord();
				cord.Load(stream);
				UpperRight = cord;
				Pos = br.ReadInt32();
				Len = br.ReadInt32();
				Sym = br.ReadInt32();
				ObjType = br.ReadByte();
				Res1 = br.ReadByte();
				Status = br.ReadByte();
				ViewType = br.ReadByte();
				Color = br.ReadInt16();
				Res2 = br.ReadInt16();
				ImpLayer = br.ReadInt16();
				Res3 = br.ReadInt16();
				return true;
			}

		}

		private class TElement {

			public Int32 Sym;
			public byte Otp;
			public byte Res0;
			public Int16 Ang;
			public Int32 nItem;
			public Int16 nText;
			public Int16 Res1;
			public Int32 Col;
			public Int16 LineWidth;
			public Int16 DiamFlags;
			public double Res2;
			public double Res3;
			public TCord[] Poly;
			//Semi-Custom
			public string ExtraString;

			public bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream, Encoding.Unicode);
				Sym = br.ReadInt32();
				Otp = br.ReadByte();
				Res0 = br.ReadByte();
				Ang = br.ReadInt16();
				nItem = br.ReadInt32();
				nText = br.ReadInt16();
				Res1 = br.ReadInt16();
				Col = br.ReadInt32();
				LineWidth = br.ReadInt16();
				DiamFlags = br.ReadInt16();
				Res2 = br.ReadDouble();
				Res3 = br.ReadDouble();
				Poly = new TCord[nItem];
				for (int i = 0; i < nItem; i++) {
					TCord cord;
					cord = new TCord();
					cord.Load(stream);
					Poly[i] = cord;
				}
				if (nText > 0) {
					char[] text = br.ReadChars(sizeof(Int32) * nText);//TCordSize*nText/2
					int pos = text.Length;
					for (int i = 0; i < text.Length; i++) {
						if (text[i] == '\0') {
							pos = i;
							break;
						}
					}
					ExtraString = new string(text, 0, pos);
				}
				return true;
			}

		}

		private class TStringIndexBlock {

			public Int32 FilePos;
			public TStringIndex[] Table = new TStringIndex[256];

			public bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream);
				FilePos = br.ReadInt32();
				for (int i = 0; i < Table.Length; i++) {
					TStringIndex index = new TStringIndex();
					index.Load(stream);
					Table[i] = index;
				}
				return true;
			}

		}

		private class TStringIndex {

			public Int32 Pos;
			public Int32 Len;
			public Int32 RecType;
			public Int32 ObjIndex;

			public bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream);
				Pos = br.ReadInt32();
				Len = br.ReadInt32();
				RecType = br.ReadInt32();
				ObjIndex = br.ReadInt32();
				return true;
			}
		
		}

		private class CustomParameterString {

			string rawString;

			Int32 type;
			public Int32 Type {
				get {
					return type;
				}
			}

			string firstField;
			public string FirstField {
				get {
					return firstField;
				}
			}

			private List<KeyValuePair<string,string>> parameters;

			public bool Load(Stream stream, TStringIndex tStringIndex) {
				if (tStringIndex.Len == 0 || tStringIndex.Pos == 0) {
					rawString = "";
					return true;
				}
				stream.Seek((long)tStringIndex.Pos, SeekOrigin.Begin);
				string str = "";
				StreamReader reader = new StreamReader(stream);
				char[] chars = new char[tStringIndex.Len];
				reader.Read(chars, 0, tStringIndex.Len);
				int i;
				for (i = 0; i < chars.Length; i++) {
					if (chars[i] == '\0') {
						break;
					}
				}
				str = new string(chars, 0, i);
				rawString = str;
				type = tStringIndex.RecType;
				parameters = new List<KeyValuePair<string, string>>();
				string[] strs = rawString.Split('\t');
				if (strs.Length == 0) {
					return true;
				}
				firstField = strs[0];
				for (int j = 1; j < strs.Length; j ++) {
					if (strs[j].Length < 2) {
						continue;
					}
					KeyValuePair<string, string> pair = new KeyValuePair<string, string>(strs[j].Substring(0,1), strs[j].Substring(1));
					parameters.Add(pair);
				}
				return true;
			}

			public string this[string key] {
				get {
					for (int i = 0; i < parameters.Count; i++) {
						if (parameters[i].Key == key) {
							return parameters[i].Value;
						}
					}
					return null;
				}
			}

		}

		private class CustomColor {

			public CustomColor() {
				isTransparent = false;
			}

			Int16 num;
			public Int16 Num {
				get {
					return num;
				}
			}

			Color color;
			public Color Color {
				get {
					return color;
				}
			}

			double c;
			double m;
			double y;
			double k;
			double t;
			bool isTransparent;

			public bool Load(CustomParameterString str) {
				if (str.Type != 9) {
					return false;
				}
				string temp;
				temp = str["n"];
				double val;
				if (temp != null) {
					num = Convert.ToInt16(temp);
				}
				temp = str["c"];
				if (temp != null) {
					val = Convert.ToDouble(temp);
					c = val;
				}
				temp = str["m"];
				if (temp != null) {
					val = Convert.ToDouble(temp);
					m = val;
				}
				temp = str["y"];
				if (temp != null) {
					val = Convert.ToDouble(temp);
					y = val;
				}
				temp = str["k"];
				if (temp != null) {
					val = Convert.ToDouble(temp);
					k = val;
				}
				temp = str["t"];
				if (temp != null) {
					val = Convert.ToDouble(temp);
					t = val;
					isTransparent = true;
				}
				color = GetColor();
				return true;
			}

			private Color GetColor() {
				Color color;
				byte a, r, g, b;
				double A, R, G, B;
				double C, M, Y, K;
				C = c / 100.0;
				M = m / 100.0;
				Y = y / 100.0;
				K = k / 100.0;
				A = t / 100.0;
				R = C * (1.0 - K) + K;
				G = M * (1.0 - K) + K;
				B = Y * (1.0 - K) + K;
				A = A * 255.0;
				R = (1.0 - R) * 255.0;
				G = (1.0 - G) * 255.0;
				B = (1.0 - B) * 255.0;
				a = (byte)Math.Round(A);
				r = (byte)Math.Round(R);
				g = (byte)Math.Round(G);
				b = (byte)Math.Round(B);
				if (isTransparent && t < 100) {
					color = Color.FromArgb(a, r, g, b);
				} else {
					color = Color.FromRgb(r, g, b);
				}
				return color;
			}

		}

	}

}